<?php
error_reporting(E_ALL); // 🚨 Exibe todos os erros (bom para desenvolvimento)
ini_set('display_errors', 1); // 🧐 Garante que os erros serão mostrados na tela
ini_set('display_startup_errors', 1); // 🚀 Exibe erros que ocorrem durante a inicialização do PHP

header('Content-Type: application/json'); // 🗣️ Diz ao navegador que a resposta é JSON

// 🎯 Caminho para o banco de dados SQLite (voltando uma pasta para encontrar 'api/max.db')
$dbPath = __DIR__ . '/../api/max.db';

try {
    // 🔗 Conecta ao banco de dados SQLite
    $db = new PDO('sqlite:' . $dbPath);
    // ⚙️ Define o modo de erro para lançar exceções em caso de problemas
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // 📩 Pega o corpo da requisição JSON enviado pelo JavaScript
    $input = file_get_contents('php://input');
    $data = json_decode($input, true); // 🧩 Decodifica o JSON em um array associativo PHP

    // ❓ Verifica se a 'action' (ação) foi especificada na requisição
    if (!isset($data['action'])) {
        echo json_encode(['success' => false, 'message' => '❌ Ação não especificada.']);
        exit;
    }

    $action = $data['action']; // 🎬 Obtém a ação a ser executada
    $response = ['success' => false, 'message' => '']; // 🔄 Inicializa a resposta padrão

    switch ($action) {
        case 'view':
            // 👀 Lógica para visualizar todos os aplicativos
            $stmt = $db->query('SELECT * FROM apps ORDER BY id DESC'); // 📊 Seleciona todos os apps
            $apps = $stmt->fetchAll(PDO::FETCH_ASSOC); // 📋 Pega todos os resultados como um array associativo
            $response = ['success' => true, 'data' => $apps]; // ✅ Sucesso e os dados dos apps
            break;

        case 'add':
            // ➕ Lógica para adicionar um novo aplicativo
            if (isset($data['name']) && isset($data['package']) && isset($data['apk_url']) && isset($data['icon_url']) && isset($data['active'])) {
                $name = trim($data['name']);
                $package = trim($data['package']);
                $apk_url = trim($data['apk_url']);
                $icon_url = trim($data['icon_url']);
                $active = (int)$data['active'];

                // ⚠️ Validação básica para campos vazios
                if (empty($name) || empty($package) || empty($apk_url) || empty($icon_url)) {
                    $response = ['success' => false, 'message' => '📝 Todos os campos são obrigatórios!'];
                    break;
                }

                // 🕵️‍♀️ Verifica se o pacote já existe para evitar duplicatas
                $stmt = $db->prepare('SELECT COUNT(*) FROM apps WHERE package = :package');
                $stmt->bindValue(':package', $package);
                $stmt->execute();
                if ($stmt->fetchColumn() > 0) {
                    $response = ['success' => false, 'message' => '⚠️ Já existe um aplicativo com este pacote.'];
                    break;
                }

                // ✍️ Prepara e executa a inserção no banco de dados
                $stmt = $db->prepare('INSERT INTO apps (name, package, apk_url, icon_url, active) VALUES (:name, :package, :apk_url, :icon_url, :active)');
                $stmt->bindValue(':name', $name);
                $stmt->bindValue(':package', $package);
                $stmt->bindValue(':apk_url', $apk_url);
                $stmt->bindValue(':icon_url', $icon_url);
                $stmt->bindValue(':active', $active);

                if ($stmt->execute()) {
                    $response = ['success' => true, 'message' => '🎉 Aplicativo adicionado com sucesso!'];
                } else {
                    $response = ['success' => false, 'message' => '🐛 Erro ao adicionar aplicativo.'];
                }
            } else {
                $response = ['success' => false, 'message' => ' incompleto para adicionar.'];
            }
            break;

        case 'edit':
            // ✏️ Lógica para editar um aplicativo existente
            if (isset($data['id']) && isset($data['name']) && isset($data['package']) && isset($data['apk_url']) && isset($data['icon_url']) && isset($data['active'])) {
                $id = (int)$data['id'];
                $name = trim($data['name']);
                $package = trim($data['package']);
                $apk_url = trim($data['apk_url']);
                $icon_url = trim($data['icon_url']);
                $active = (int)$data['active'];

                // ⚠️ Validação básica para campos vazios
                if (empty($name) || empty($package) || empty($apk_url) || empty($icon_url)) {
                    $response = ['success' => false, 'message' => '📝 Todos os campos são obrigatórios!'];
                    break;
                }

                // 🕵️‍♀️ Verifica se o pacote já existe para outro ID (evita conflitos ao editar)
                $stmt = $db->prepare('SELECT COUNT(*) FROM apps WHERE package = :package AND id != :id');
                $stmt->bindValue(':package', $package);
                $stmt->bindValue(':id', $id);
                $stmt->execute();
                if ($stmt->fetchColumn() > 0) {
                    $response = ['success' => false, 'message' => '⚠️ Já existe outro aplicativo com este pacote.'];
                    break;
                }

                // ✍️ Prepara e executa a atualização no banco de dados
                $stmt = $db->prepare('UPDATE apps SET name = :name, package = :package, apk_url = :apk_url, icon_url = :icon_url, active = :active WHERE id = :id');
                $stmt->bindValue(':name', $name);
                $stmt->bindValue(':package', $package);
                $stmt->bindValue(':apk_url', $apk_url);
                $stmt->bindValue(':icon_url', $icon_url);
                $stmt->bindValue(':active', $active);
                $stmt->bindValue(':id', $id);

                if ($stmt->execute()) {
                    $response = ['success' => true, 'message' => '✅ Aplicativo atualizado com sucesso!'];
                } else {
                    $response = ['success' => false, 'message' => '🐛 Erro ao atualizar aplicativo.'];
                }
            } else {
                $response = ['success' => false, 'message' => ' incomplete para editar.'];
            }
            break;

        case 'delete':
            // 🗑️ Lógica para excluir um aplicativo
            if (isset($data['id'])) {
                $id = (int)$data['id'];
                $stmt = $db->prepare('DELETE FROM apps WHERE id = :id'); // 🚮 Prepara a exclusão
                $stmt->bindValue(':id', $id);

                if ($stmt->execute()) {
                    $response = ['success' => true, 'message' => '🔥 Aplicativo excluído com sucesso!'];
                } else {
                    $response = ['success' => false, 'message' => '🐛 Erro ao excluir aplicativo.'];
                }
            } else {
                $response = ['success' => false, 'message' => 'ID do aplicativo não especificado para exclusão.'];
            }
            break;

        default:
            $response = ['success' => false, 'message' => '🚫 Ação inválida.'];
            break;
    }

    echo json_encode($response); // 📦 Envia a resposta JSON de volta ao navegador

} catch (PDOException $e) {
    // 🛑 Captura e exibe erros específicos do banco de dados
    echo json_encode(['success' => false, 'message' => '🚨 Erro no banco de dados: ' . $e->getMessage()]);
} catch (Exception $e) {
    // 🚨 Captura outras exceções gerais e inesperadas
    echo json_encode(['success' => false, 'message' => '🚫 Ocorreu um erro inesperado: ' . $e->getMessage()]);
}
?>