<?php
// Configurações para remover todas as restrições - Mantenha estas!
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/php_errors.log'); // Certifique-se que este arquivo é gravável
ini_set('upload_max_filesize', '128M'); // Aumentei para um valor explícito, mas -1 também funciona se o PHP permitir
ini_set('post_max_size', '128M');      // Deve ser maior ou igual a upload_max_filesize
ini_set('max_input_time', '-1');
ini_set('max_execution_time', '0');
ini_set('memory_limit', '512M'); // Aumentado para lidar com APKs maiores e a biblioteca

require 'libs/php-apk-parser/vendor/autoload.php';

$static_url = dirname($_SERVER['SCRIPT_NAME']) . '/assets/';
// Inclua session_check.php aqui, mas certifique-se que ele não gera NENHUMA SAÍDA
// antes dos headers ou do JSON, especialmente em requisições POST.
include 'session_check.php'; 

// Configurações de upload
$uploadDir = __DIR__ . '/uploads/';
$iconDir = $uploadDir . 'icons/';
$apkDir = $uploadDir . 'apks/';

// URL base
$basePath = dirname($_SERVER['SCRIPT_NAME']);
// Garante que o protocolo e o host estão corretos para as URLs
$protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https://" : "http://";
$baseUrl = $protocol . $_SERVER['HTTP_HOST'] . $basePath;

// Criar diretórios se não existirem
// Verifique as permissões de escrita para o usuário do servidor web!
if (!file_exists($iconDir)) {
    mkdir($iconDir, 0777, true);
    error_log("Diretório de ícones criado: " . $iconDir);
}
if (!file_exists($apkDir)) {
    mkdir($apkDir, 0777, true);
    error_log("Directorio de Apks Creadas: " . $apkDir);
}

$dbPath = __DIR__ . '/api/max.db';
try {
    $db = new PDO('sqlite:' . $dbPath);
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    // Cria a tabela se ela não existir (boa prática)
    $db->exec("CREATE TABLE IF NOT EXISTS apps (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL,
        package TEXT NOT NULL UNIQUE,
        apk_url TEXT NOT NULL,
        icon_url TEXT NOT NULL,
        active INTEGER DEFAULT 1
    )");
} catch (PDOException $e) {
    error_log("Erro ao conectar/criar DB: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => '❌ Error no hay Banco de Datos: ' . $e->getMessage()]);
    exit;
}


// Função para extrair o nome do pacote do APK
function getApkPackageName($filePath) {
    error_log("Tentando analisar APK: " . $filePath);
    try {
        if (!file_exists($filePath)) {
            throw new Exception("No se encontr el archivo APK.: " . $filePath);
        }
        
        $apk = new \ApkParser\Parser($filePath);
        $packageName = $apk->getManifest()->getPackageName();
        error_log("Paquete APK extrado: " . $packageName);
        return $packageName;
    } catch (Exception $e) {
        error_log("Error crtico al analizar el APK: " . $e->getMessage());
        return null;
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // É crucial que NADA seja enviado para o navegador antes destas linhas
    header('Content-Type: application/json');
    header('Access-Control-Allow-Origin: *'); // Considere restringir em produção

    error_log("Se recibi una solicitud POST. Accin: " . ($_POST['action'] ?? 'N/A'));

    try {
        $action = $_POST['action'] ?? null;
        $id = isset($_POST['id']) ? (int)$_POST['id'] : null;
        $name = isset($_POST['name']) ? trim($_POST['name']) : null;
        $active = isset($_POST['active']) ? (int)$_POST['active'] : 0;

        // Processar uploads
        $apkUrl = null;
        $iconUrl = null;
        $package = null;

        if ($action === 'add' || $action === 'edit') {
            if (empty($name)) {
                throw new Exception('El nombre de la aplicacin es obligatorio..');
            }

            // Processar upload do APK
            if (isset($_FILES['apk_file']) && $_FILES['apk_file']['error'] === UPLOAD_ERR_OK) {
                // Verificação de tipo de arquivo mais robusta
                $finfo = finfo_open(FILEINFO_MIME_TYPE);
                $mime = finfo_file($finfo, $_FILES['apk_file']['tmp_name']);
                finfo_close($finfo);
                
                // Tipos MIME comuns para APK, incluindo a detecção genérica para casos onde o servidor não mapeia corretamente
                $allowedMimeTypes = [
                    'application/vnd.android.package-archive',
                    'application/octet-stream', // Em alguns casos, pode ser detectado genericamente
                    'application/zip'           // APKs são, de fato, arquivos zip
                ];

                if (!in_array($mime, $allowedMimeTypes)) {
                    error_log("Tipo MIME no vlido para APK: " . $mime);
                    throw new Exception('Por favor, enve un archivo APK vlido.: ' . $mime);
                }

                $apkExt = strtolower(pathinfo($_FILES['apk_file']['name'], PATHINFO_EXTENSION));
                if ($apkExt !== 'apk') {
                     error_log("Extensin de archivo APK no vlida: " . $apkExt);
                     throw new Exception('La extensin del archivo debe ser .apk');
                }

                $apkFilename = uniqid('apk_') . '.' . $apkExt;
                $apkPath = $apkDir . $apkFilename;

                if (move_uploaded_file($_FILES['apk_file']['tmp_name'], $apkPath)) {
                    $apkUrl = $baseUrl . '/uploads/apks/' . $apkFilename;
                    $package = getApkPackageName($apkPath);
                    if ($package === null) {
                         // Se o pacote não pôde ser extraído, delete o APK e avise
                         unlink($apkPath);
                         throw new Exception('No pudimos extraer el nombre del paquete del APK. Archivo no vlido o daado.');
                    }
                    error_log("El APK se movi y el paquete se extrajo correctamente.. URL: " . $apkUrl . ", Pacote: " . $package);
                } else {
                    $uploadError = $_FILES['apk_file']['error'];
                    error_log("No se pudo mover el archivo APK. Cdigo de error: " . $uploadError);
                    throw new Exception("No se pudo mover el archivo APK. Error de carga.: " . $uploadError);
                }
            } elseif ($action === 'add' && (!isset($_FILES['apk_file']) || $_FILES['apk_file']['error'] !== UPLOAD_ERR_OK)) {
                 throw new Exception('No se ha subido ningn archivo APK vlido para la nueva aplicacin.');
            } elseif ($action === 'editar' && (!isset($_FILES['apk_file']) || $_FILES['apk_file']['error'] === UPLOAD_ERR_NO_FILE)) {
                // Manter o APK existente se não for enviado um novo durante a edição
                $stmt = $db->prepare("SELECT apk_url, package FROM apps WHERE id = ?");
                $stmt->execute([$id]);
                $appData = $stmt->fetch(PDO::FETCH_ASSOC);
                $apkUrl = $appData['apk_url'] ?? null;
                $package = $appData['package'] ?? null;
                error_log("Edicin: No se ha subido ningn nuevo APK. La URL se mantiene.: " . $apkUrl . ", Paquete: " . $package);
            }


            // Processar upload do ícone
            if (isset($_FILES['icon_file']) && $_FILES['icon_file']['error'] === UPLOAD_ERR_OK) {
                $iconExt = strtolower(pathinfo($_FILES['icon_file']['name'], PATHINFO_EXTENSION));
                $allowedIconExts = ['jpg', 'jpeg', 'png', 'gif', 'svg'];
                if (!in_array($iconExt, $allowedIconExts)) {
                    error_log("Extensin de icono no vlida: " . $iconExt);
                    throw new Exception('Forma de icono no vlida. Use JPG, PNG, GIF ou SVG.');
                }

                $iconFilename = uniqid('icon_') . '.' . $iconExt;
                $iconPath = $iconDir . $iconFilename;

                if (move_uploaded_file($_FILES['icon_file']['tmp_name'], $iconPath)) {
                    $iconUrl = $baseUrl . '/uploads/icons/' . $iconFilename;
                    error_log("El icono se movi con xito. URL: " . $iconUrl);
                } else {
                    $uploadError = $_FILES['icon_file']['error'];
                    error_log("Error al cargar el archivo de icono. Error de Codigo: " . $uploadError);
                    throw new Exception('Falha no upload do arquivo de ícone. Erro de upload: ' . $uploadError);
                }
            } elseif ($action === 'add' && (!isset($_FILES['icon_file']) || $_FILES['icon_file']['error'] !== UPLOAD_ERR_OK)) {
                 throw new Exception('No se ha subido ningn archivo de icono vlido para la nueva aplicacin.');
            } elseif ($action === 'edit' && (!isset($_FILES['icon_file']) || $_FILES['icon_file']['error'] === UPLOAD_ERR_NO_FILE)) {
                // Manter o ícone existente se não for enviado um novo durante a edição
                $stmt = $db->prepare("SELECT icon_url FROM apps WHERE id = ?");
                $stmt->execute([$id]);
                $iconUrl = $stmt->fetchColumn() ?? null;
                error_log("Editado: No se han enviado nuevos iconos. Se mantiene la URL: " . $iconUrl);
            }
            
            // Validações finais antes de salvar
            if (empty($apkUrl) || empty($iconUrl) || empty($package)) {
                 throw new Exception('No se pudieron determinar las URL del APK/icono o paquete. Por favor, revise las cargas y la extraccin del paquete.');
            }


            if ($action === 'add') {
                $stmt = $db->prepare("INSERT INTO apps (name, package, apk_url, icon_url, active) VALUES (?, ?, ?, ?, ?)");
                $stmt->execute([$name, $package, $apkUrl, $iconUrl, $active]);
                error_log("Aplicacin aadida a la base de datos.: " . $name);
                echo json_encode(['status' => 'success', 'message' => '✅ Aplicacin aadida correctamente']);
                exit;
            } elseif ($action === 'edit') {
                $stmt = $db->prepare("UPDATE apps SET name = ?, package = COALESCE(?, package), apk_url = COALESCE(?, apk_url), icon_url = COALESCE(?, icon_url), active = ? WHERE id = ?");
                $stmt->execute([$name, $package, $apkUrl, $iconUrl, $active, $id]);
                error_log("Aplicacin actualizada en la base de datos.: " . $name . " (ID: " . $id . ")");
                echo json_encode(['status' => 'success', 'message' => '🔄La aplicacin se actualiz correctamente.!']);
                exit;
            }
        } elseif ($action === 'delete') {
            error_log("Intentando eliminar la aplicacin con ID: " . $id);
            // Obter URLs antes de deletar
            $stmt = $db->prepare("SELECT apk_url, icon_url FROM apps WHERE id = ?");
            $stmt->execute([$id]);
            $app = $stmt->fetch(PDO::FETCH_ASSOC);

            // Deletar do banco de dados
            $stmt = $db->prepare("DELETE FROM apps WHERE id = ?");
            $stmt->execute([$id]);
            error_log("Aplicacin eliminada de la base de datos. (ID: " . $id . ")");

            // Deletar arquivos físicos
            if ($app && $app['apk_url']) {
                $relativePath = str_replace($baseUrl, '', $app['apk_url']);
                $filePath = __DIR__ . '/' . ltrim($relativePath, '/');
                if (file_exists($filePath)) {
                    unlink($filePath);
                    error_log("APK Borrado: " . $filePath);
                } else {
                    error_log("No se encontr el archivo APK para la seleccin.: " . $filePath);
                }
            }
            if ($app && $app['icon_url']) {
                $relativePath = str_replace($baseUrl, '', $app['icon_url']);
                $filePath = __DIR__ . '/' . ltrim($relativePath, '/');
                if (file_exists($filePath)) {
                    unlink($filePath);
                    error_log("Ícone deletado: " . $filePath);
                } else {
                    error_log("Ícone não encontrado para deleção: " . $filePath);
                }
            }

            echo json_encode(['status' => 'success', 'message' => '🗑️ App deletado com sucesso!']);
            exit;
        } elseif ($action === 'toggle') {
            error_log("Tentando alternar status para ID: " . $id);
            // Obter e inverter o status
            $stmt = $db->prepare("SELECT active FROM apps WHERE id = ?");
            $stmt->execute([$id]);
            $currentStatus = $stmt->fetchColumn();
            $newStatus = $currentStatus ? 0 : 1;

            $stmt = $db->prepare("UPDATE apps SET active = ? WHERE id = ?");
            $stmt->execute([$newStatus, $id]);
            error_log("Status do app (ID: " . $id . ") alterado para: " . $newStatus);

            echo json_encode([
                'status' => 'success',
                'message' => '🔄 Status do app alterado com sucesso!',
                'newStatus' => $newStatus
            ]);
            exit;
        }

        echo json_encode(['status' => 'error', 'message' => 'Ação desconhecida']);
        exit;

    } catch (Exception $e) {
        error_log("Exceção capturada na requisição POST: " . $e->getMessage());
        http_response_code(500); // Garante que o status code 500 seja enviado
        echo json_encode(['status' => 'error', 'message' => '❌ Erro: ' . $e->getMessage()]);
        exit;
    }
}

// Lógica para exibir a tabela de apps (GET request)
$apps = $db->query("SELECT * FROM apps ORDER BY name")->fetchAll(PDO::FETCH_ASSOC);

$page_title = "📱 Gerenciamento de Apps";

$page_content = '
<div class="card">
    <div class="card-header border-0 pt-6">
        <div class="card-title">
            <div class="d-flex align-items-center position-relative my-1">
                <i class="ki-outline ki-magnifier fs-3 position-absolute ms-5"></i>
                <input type="text" id="search_apps" class="form-control form-control-solid w-250px ps-12" placeholder="🔍 Pesquisar Apps" />
            </div>
        </div>
        <div class="card-toolbar">
            <div class="d-flex justify-content-end">
                <button type="button" class="btn btn-primary" onclick="openAddModal()">➕ Adicionar App</button>
            </div>
        </div>
    </div>
    <div class="card-body pt-0">
        <table class="table align-middle table-row-dashed fs-6 gy-5" id="apps_table">
            <thead>
                <tr class="text-start text-gray-500 fw-bold fs-7 text-uppercase gs-0">
                    <th class="min-w-50px">🖼️ Icono</th>
                    <th class="min-w-150px">📛 Nombre</th>
                    <th class="min-w-150px">📦 Paquete</th>
                    <th class="min-w-100px">🔘 Status</th>
                    <th class="min-w-100px">🛠️ Comportamiento</th>
                </tr>
            </thead>
            <tbody id="apps_table_body" class="fw-semibold text-gray-600">
';

foreach ($apps as $app) {
    $statusClass = $app['active'] ? 'badge-light-success' : 'badge-light-danger';
    $statusText = $app['active'] ? '✅ Ativo' : '⛔ Inativo';

    // Garante que a URL do ícone tenha o domínio completo para exibição
    $iconUrlDisplay = $app['icon_url'];
    if ($iconUrlDisplay && !preg_match('/^http(s)?:\/\//', $iconUrlDisplay)) {
        // Se a URL não for absoluta, torna-a absoluta usando o baseUrl
        $iconUrlDisplay = $baseUrl . '/' . ltrim($iconUrlDisplay, '/');
    }
    // Fallback para ícone padrão se não houver URL ou estiver quebrada
    $iconUrlDisplay = $iconUrlDisplay ?: $static_url . 'media/icons/duotune/general/gen006.svg';

    $page_content .= "<tr data-id='{$app['id']}'>
        <td><img src='{$iconUrlDisplay}' alt='{$app['name']}' style='width: 40px; height: 40px; object-fit: contain;' onerror=\"this.src='{$static_url}media/icons/duotune/general/gen006.svg'\"></td>
        <td>{$app['name']}</td>
        <td>
            <div class='d-flex align-items-center'>
                <span class='me-2'>{$app['package']}</span>
                <button class='btn btn-sm btn-icon btn-light-primary' onclick=\"copyToClipboard('{$app['package']}')\">
                    <i class='ki-outline ki-copy fs-2'></i>
                </button>
            </div>
        </td>
        <td>
            <div class='d-flex align-items-center'>
                <div class='form-check form-switch form-check-custom form-check-solid me-3'>
                    <input class='form-check-input status-toggle' type='checkbox' data-id='{$app['id']}' " . ($app['active'] ? 'checked' : '') . "/>
                </div>
                <span class='badge {$statusClass}'>{$statusText}</span>
            </div>
        </td>
        <td>
            <button class='btn btn-sm btn-light-primary' onclick=\"editApp({$app['id']}, '{$app['name']}', {$app['active']})\">✏️ Editar</button>
            <button class='btn btn-sm btn-light-danger' onclick=\"deleteApp({$app['id']}, '{$app['name']}')\">🗑️ Excluir</button>
        </td>
    </tr>";
}

$page_content .= '
            </tbody>
        </table>
    </div>
</div>

<div class="modal fade" id="app_modal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered mw-650px">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="modal_title" class="fw-bold">➕ Adicionar App</h2>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="app_form" enctype="multipart/form-data">
                    <input type="hidden" id="app_id">
                    <div class="mb-7">
                        <label for="app_name" class="fs-6 fw-semibold mb-2">📛 Nome do App</label>
                        <input type="text" class="form-control" id="app_name" placeholder="Digite o nome do app" required>
                    </div>
                    <div class="mb-7">
                        <label class="fs-6 fw-semibold mb-2">📦 Nome do Pacote</label>
                        <div class="alert alert-info">
                            <i class="ki-outline ki-information fs-3 me-2"></i>
                            O nome do pacote será extraído automaticamente do arquivo APK
                        </div>
                    </div>
                    <div class="mb-7">
                        <label for="apk_file" class="fs-6 fw-semibold mb-2">📲 Arquivo APK</label>
                        <input type="file" class="form-control" id="apk_file" accept=".apk">
                        <div class="form-text">Tamanho máximo: 128MB</div>
                        <div id="current_apk" class="mt-2"></div>
                    </div>
                    <div class="mb-7">
                        <label for="icon_file" class="fs-6 fw-semibold mb-2">🖼️ Ícone do App</label>
                        <input type="file" class="form-control" id="icon_file" accept="image/png, image/jpeg, image/gif, image/svg+xml">
                        <div class="form-text">Tamanho recomendado: 512x512 pixels</div>
                        <div id="current_icon" class="mt-2"></div>
                    </div>
                    <div class="mb-7">
                        <div class="form-check form-switch form-check-custom form-check-solid">
                            <input class="form-check-input" type="checkbox" id="app_active" value="1" checked>
                            <label class="form-check-label" for="app_active">🔘 Ativar App</label>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-light" data-bs-dismiss="modal">❌ Cancelar</button>
                <button type="button" class="btn btn-primary" onclick="saveApp()">💾 Salvar</button>
            </div>
        </div>
    </div>
</div>
';

include 'includes/layout.php';
?>

---

```javascript
<script>
document.addEventListener('DOMContentLoaded', function() {
    let appsTable = $('#apps_table').DataTable({
        searching: true,
        paging: true,
        ordering: true,
        info: true,
        responsive: false,
        language: {
            url: '//cdn.datatables.net/plug-ins/1.13.6/i18n/pt-BR.json'
        }
    });

    document.getElementById("search_apps").addEventListener("input", function() {
        appsTable.search(this.value).draw();
    });

    // Delegação de eventos para os toggles de status
    document.getElementById('apps_table').addEventListener('change', function(e) {
        if (e.target.classList.contains('status-toggle')) {
            const id = e.target.getAttribute('data-id');
            const isChecked = e.target.checked;
            toggleStatus(id, isChecked, e.target);
        }
    });

    window.copyToClipboard = function(text) {
        navigator.clipboard.writeText(text).then(() => {
            Swal.fire({
                icon: 'success',
                title: '📋 Copiado!',
                text: 'Nome do pacote copiado para a área de transferência',
                timer: 1000,
                showConfirmButton: false
            });
        }).catch(err => {
            console.error('Erro ao copiar para a área de transferência:', err);
            Swal.fire({
                icon: 'error',
                title: 'Erro!',
                text: 'Falha ao copiar o nome do pacote.',
            });
        });
    };

    window.openAddModal = function() {
        document.getElementById('modal_title').textContent = '➕ Adicionar App';
        document.getElementById('app_id').value = '';
        document.getElementById('app_name').value = '';
        // Para adicionar, ambos os arquivos são requeridos via JS, mas o backend também valida
        document.getElementById('apk_file').required = true;
        document.getElementById('icon_file').required = true;
        document.getElementById('apk_file').value = ''; // Limpa o input file
        document.getElementById('icon_file').value = ''; // Limpa o input file
        document.getElementById('app_active').checked = true;
        document.getElementById('current_apk').innerHTML = ''; // Limpa exibição de APK atual
        document.getElementById('current_icon').innerHTML = ''; // Limpa exibição de ícone atual
        $('#app_modal').modal('show');
    };

    window.editApp = function(id, name, active) {
        document.getElementById('modal_title').textContent = '✏️ Editar App';
        document.getElementById('app_id').value = id;
        document.getElementById('app_name').value = name;
        document.getElementById('app_active').checked = Boolean(active);
        // Para editar, arquivos não são requeridos se não forem trocados
        document.getElementById('apk_file').required = false;
        document.getElementById('icon_file').required = false;
        document.getElementById('apk_file').value = '';
        document.getElementById('icon_file').value = '';

        // Mostrar arquivos atuais buscando detalhes do servidor
        fetch('get_app_details.php?id=' + id) // Assumindo que você tem um get_app_details.php para buscar info
            .then(response => {
                if (!response.ok) {
                    throw new Error(`Erro ${response.status}: ${response.statusText}`);
                }
                return response.json();
            })
            .then(data => {
                if (!data) throw new Error('Resposta vazia do servidor ao buscar detalhes.');

                if (data.apk_url) {
                    const apkName = data.apk_url.split('/').pop();
                    document.getElementById('current_apk').innerHTML =
                        `<div class="alert alert-info p-2">
                            <i class="ki-outline ki-file fs-3 me-2"></i>
                            APK atual: ${apkName}
                        </div>`;
                } else {
                    document.getElementById('current_apk').innerHTML = ''; // Limpa se não houver
                }

                if (data.icon_url) {
                    document.getElementById('current_icon').innerHTML =
                        `<div class="alert alert-info p-2">
                            <img src="${data.icon_url}" alt="Ícone atual" style="width: 30px; height: 30px; object-fit: contain;" class="me-2">
                            Ícone atual
                        </div>`;
                } else {
                    document.getElementById('current_icon').innerHTML = ''; // Limpa se não houver
                }
            })
            .catch(error => {
                console.error('Erro ao carregar detalhes do app para edição:', error);
                Swal.fire({
                    icon: 'error',
                    title: 'Erro!',
                    text: 'Falha ao carregar detalhes do app para edição.',
                });
                document.getElementById('current_apk').innerHTML = '';
                document.getElementById('current_icon').innerHTML = '';
            });

        $('#app_modal').modal('show');
    };

    window.saveApp = function() {
        const id = document.getElementById('app_id').value;
        const name = document.getElementById('app_name').value.trim();
        const active = document.getElementById('app_active').checked ? 1 : 0;
        const apkFile = document.getElementById('apk_file').files[0];
        const iconFile = document.getElementById('icon_file').files[0];

        // Validar campos obrigatórios
        if (!name) {
            Swal.fire({
                icon: 'error',
                title: '❌ Erro',
                text: 'Nome do app é obrigatório!',
            });
            return;
        }

        // Para adição, ambos os arquivos são obrigatórios
        if (!id && (!apkFile || !iconFile)) {
            Swal.fire({
                icon: 'error',
                title: '❌ Erro',
                text: 'Arquivos APK e Ícone são obrigatórios para novos apps!',
            });
            return;
        }

        // Configurar o botão de loading
        const saveBtn = document.querySelector('#app_modal .btn-primary');
        const originalText = saveBtn.innerHTML;
        saveBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> Salvando...';
        saveBtn.disabled = true;

        // Criar FormData
        const formData = new FormData();
        formData.append('action', id ? 'edit' : 'add');
        if (id) formData.append('id', id);
        formData.append('name', name);
        formData.append('active', active);
        if (apkFile) formData.append('apk_file', apkFile);
        if (iconFile) formData.append('icon_file', iconFile);

        // Enviar requisição
        fetch(window.location.href, { // Envia para o próprio script PHP
            method: 'POST',
            body: formData,
        })
        .then(response => {
            // Se a resposta não for OK (2xx), tente ler como texto para depuração
            if (!response.ok) {
                 return response.text().then(text => {
                    console.error('Resposta de erro do servidor (texto):', text);
                    throw new Error(`Erro HTTP! Status: ${response.status}. Detalhes: ${text.substring(0, 200)}...`); // Limita para não sobrecarregar
                 });
            }
            return response.json();
        })
        .then(data => {
            if (!data) {
                throw new Error('Resposta vazia ou inválida do servidor.');
            }

            if (data.status === 'success') {
                Swal.fire({
                    icon: 'success',
                    title: '✅ Sucesso!',
                    text: data.message,
                }).then(() => {
                    $('#app_modal').modal('hide');
                    location.reload(); // Recarrega a página para atualizar a tabela
                });
            } else {
                // Se o status for 'error' mas a resposta JSON é válida
                throw new Error(data.message || 'Erro desconhecido ao salvar.');
            }
        })
        .catch(error => {
            console.error('Erro na requisição AJAX:', error);
            Swal.fire({
                icon: 'error',
                title: '❌ Erro',
                text: error.message || 'Falha ao salvar. Verifique o console do navegador e os logs do servidor para detalhes.',
            });
        })
        .finally(() => {
            saveBtn.disabled = false;
            saveBtn.innerHTML = originalText;
        });
    };

    window.deleteApp = function(id, name) {
        Swal.fire({
            title: '⚠️ Tem certeza?',
            text: `Você está prestes a excluir o app: ${name}. Isso também removerá os arquivos APK e ícone.`,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#d33',
            cancelButtonColor: '#3085d6',
            confirmButtonText: '🗑️ Sim, excluir!',
            cancelButtonText: '❌ Cancelar'
        }).then((result) => {
            if (result.isConfirmed) {
                fetch(window.location.href, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded', // FormData é melhor para arquivos, mas para delete, URLSearchParams serve
                    },
                    body: new URLSearchParams({
                        action: 'delete',
                        id: id,
                    }),
                })
                .then(response => {
                    if (!response.ok) {
                        return response.text().then(text => {
                            console.error('Resposta de erro do servidor (texto):', text);
                            throw new Error(`Erro HTTP! Status: ${response.status}. Detalhes: ${text.substring(0, 200)}...`);
                        });
                    }
                    return response.json();
                })
                .then(data => {
                    if (data.status === 'success') {
                        Swal.fire({
                            icon: 'success',
                            title: '🗑️ Excluído!',
                            text: data.message,
                        }).then(() => location.reload());
                    } else {
                        throw new Error(data.message || 'Erro ao excluir');
                    }
                })
                .catch(error => {
                    console.error('Erro na requisição AJAX de exclusão:', error);
                    Swal.fire({
                        icon: 'error',
                        title: '❌ Erro',
                        text: error.message || 'Falha ao excluir. Verifique o console para detalhes.',
                    });
                });
            }
        });
    };

    window.toggleStatus = function(id, isChecked, checkbox) {
        const row = checkbox.closest('tr');
        const statusBadge = row.querySelector('.badge');

        // Desabilitar o checkbox durante a requisição
        checkbox.disabled = true;

        fetch(window.location.href, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: new URLSearchParams({
                action: 'toggle',
                id: id,
            }),
        })
        .then(response => {
            if (!response.ok) {
                 return response.text().then(text => {
                    console.error('Resposta de erro do servidor (texto):', text);
                    throw new Error(`Erro HTTP! Status: ${response.status}. Detalhes: ${text.substring(0, 200)}...`);
                 });
            }
            return response.json();
        })
        .then(data => {
            if (data.status !== 'success') {
                throw new Error(data.message || 'Falha ao alterar status');
            }

            // Atualizar visualização apenas após sucesso no servidor
            if (isChecked) {
                statusBadge.className = 'badge badge-light-success';
                statusBadge.textContent = '✅ Ativo';
            } else {
                statusBadge.className = 'badge badge-light-danger';
                statusBadge.textContent = '⛔ Inativo';
            }
        })
        .catch(error => {
            console.error('Erro na requisição AJAX de toggle:', error);
            // Reverter o estado do checkbox em caso de erro
            checkbox.checked = !isChecked;
            Swal.fire({
                icon: 'error',
                title: '❌ Erro',
                text: error.message || 'Falha ao alterar status',
            });
        })
        .finally(() => {
            checkbox.disabled = false;
        });
    };
});
</script>