<?php
require_once __DIR__ . '/config.php';

header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

function self_url(): string {
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $path = $_SERVER['SCRIPT_NAME'] ?? '/gateway.php';
    return $scheme . '://' . $host . $path;
}

function cache_path(): string {
    return __DIR__ . '/dns_cache.json';
}

function read_cache(): array {
    $path = cache_path();
    if (!file_exists($path)) return [];
    $j = json_decode(@file_get_contents($path), true);
    return is_array($j) ? $j : [];
}

function write_cache(array $cache): void {
    $path = cache_path();
    $tmp = $path . '.tmp';
    @file_put_contents($tmp, json_encode($cache, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES), LOCK_EX);
    @rename($tmp, $path);
}

function http_get(string $url, int $timeout = 6): array {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_CONNECTTIMEOUT => $timeout,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_USERAGENT => 'Neutro7-MultiDNS/1.0'
    ]);
    $body = curl_exec($ch);
    $err  = curl_error($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
    curl_close($ch);
    return [$code, $body === false ? '' : $body, $err];
}

function auth_ok(string $json): bool {
    $d = json_decode($json, true);
    if (!is_array($d)) return false;
    if (isset($d['user_info']['auth'])) return (string)$d['user_info']['auth'] === '1' || $d['user_info']['auth'] === 1;
    if (isset($d['auth'])) return (string)$d['auth'] === '1' || $d['auth'] === 1;
    if (isset($d['user_info']['status'])) return true;
    return false;
}

function check_dns_login(string $dns, string $u, string $p): bool {
    $dns = rtrim(trim($dns), '/');
    if ($dns === '') return false;
    $url = $dns . '/player_api.php?username=' . rawurlencode($u) . '&password=' . rawurlencode($p);
    [$code, $body, ] = http_get($url, 6);
    if ($code < 200 || $code >= 300) return false;
    return auth_ok($body);
}

/* =========================
   STRICT DNS RESOLVER FIX
   ========================= */

function resolve_dns(string $u, string $p, array $dns_list, array &$tested): ?string {
    $tested = [];

    // 1. HARD BLOCK if NO DNS in panel
    if (count($dns_list) === 0) {
        $cache = read_cache();
        if (isset($cache[$u])) {
            unset($cache[$u]);
            write_cache($cache);
        }
        return null;
    }

    $cache = read_cache();
    $cached = $cache[$u]['dns'] ?? '';

    // 2. If cached DNS was DELETED from panel -> invalidate it
    if (is_string($cached) && $cached !== '') {
        if (!in_array($cached, $dns_list, true)) {
            unset($cache[$u]);
            write_cache($cache);
            $cached = '';
        }
    }

    // 3. Try cached DNS ONLY if still allowed by panel
    if (is_string($cached) && $cached !== '') {
        $tested[] = $cached;
        if (check_dns_login($cached, $u, $p)) {
            return $cached;
        }
    }

    // 4. Scan CURRENT dns.json only
    foreach ($dns_list as $dns) {
        if (!is_string($dns)) continue;
        $dns = trim($dns);
        if ($dns === '' || $dns === $cached) continue;
        $tested[] = $dns;
        if (check_dns_login($dns, $u, $p)) {
            $cache[$u] = ['dns' => $dns, 'ts' => time()];
            write_cache($cache);
            return $dns;
        }
    }

    // 5. Nothing worked -> clear cache and fail
    if (isset($cache[$u])) {
        unset($cache[$u]);
        write_cache($cache);
    }

    return null;
}

$u = isset($_GET['username']) ? (string)$_GET['username'] : '';
$p = isset($_GET['password']) ? (string)$_GET['password'] : '';

if ($u === '' || $p === '') {
    header('Content-Type: text/plain; charset=UTF-8');
    echo self_url();
    exit;
}

$dns_list = read_dns_list();
$tested = [];
$dns = resolve_dns($u, $p, $dns_list, $tested);

header('Content-Type: application/json; charset=UTF-8');

if ($dns === null) {
    http_response_code(403);
    echo json_encode([
        'status' => 'error',
        'error' => 'No active DNS in panel',
        'dns_tested' => $tested
    ], JSON_UNESCAPED_SLASHES);
    exit;
}

echo json_encode([
    'status' => 'ok',
    'dns' => rtrim($dns, '/')
], JSON_UNESCAPED_SLASHES);
