<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

$static_url = dirname($_SERVER['SCRIPT_NAME']) . '/rtx/';
include 'session_check.php';

// Caminhos
$apiDir  = __DIR__ . '/api';
$dbPath  = $apiDir . '/express.db';
$apksDir = $apiDir . '/apks';

// Cria pastas se não existirem
if (!is_dir($apiDir))  @mkdir($apiDir, 0755, true);
if (!is_dir($apksDir)) @mkdir($apksDir, 0755, true);

// Helpers
function base_url_for_dir(string $dirRelativeToDocRoot): string {
    $isHttps = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
        || (isset($_SERVER['SERVER_PORT']) && (int)$_SERVER['SERVER_PORT'] === 443);

    $scheme = $isHttps ? 'https' : 'http';
    $host   = $_SERVER['HTTP_HOST'] ?? 'localhost';

    $dir = rtrim($dirRelativeToDocRoot, '/');
    if ($dir === '/' || $dir === '\\') $dir = '';
    return $scheme . '://' . $host . $dir;
}

function project_base_dir_from_this_script(): string {
    $dir = dirname($_SERVER['SCRIPT_NAME'] ?? '/');
    $dir = str_replace('\\', '/', $dir);
    if ($dir === '.') $dir = '/';
    if ($dir === '/') return '';
    return rtrim($dir, '/');
}

function h($s) { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

$message = '';
$error = '';

try {
    $pdo = new PDO('sqlite:' . $dbPath);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Tabela de update
    $pdo->exec("
        CREATE TABLE IF NOT EXISTS update_settings (
            id INTEGER PRIMARY KEY,
            version_code INTEGER NOT NULL DEFAULT 0,
            version_name TEXT DEFAULT '',
            apk_source TEXT NOT NULL DEFAULT 'remote',
            apk_url TEXT DEFAULT '',
            local_file TEXT DEFAULT '',
            changelog TEXT DEFAULT '',
            force INTEGER NOT NULL DEFAULT 0,
            updated_at TEXT DEFAULT ''
        )
    ");

    // Garante linha id=1
    $pdo->exec("INSERT OR IGNORE INTO update_settings (id) VALUES (1)");

    // Salvar configurações
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $version_code = isset($_POST['version_code']) ? (int)$_POST['version_code'] : 0;
        $version_name = trim($_POST['version_name'] ?? '');
        $apk_source   = ($_POST['apk_source'] ?? 'remote') === 'local' ? 'local' : 'remote';
        $apk_url      = trim($_POST['apk_url'] ?? '');
        $changelog    = trim($_POST['changelog'] ?? '');
        $force        = isset($_POST['force']) ? 1 : 0;

        if ($version_code <= 0) {
            throw new Exception("version_code inválido. Use um número maior que 0.");
        }

        $local_file = '';

        if ($apk_source === 'remote') {
            if (empty($apk_url) || !filter_var($apk_url, FILTER_VALIDATE_URL)) {
                throw new Exception("URL remota do APK inválida.");
            }
        } else {
            // LOCAL: upload do APK
            if (!isset($_FILES['apk_file']) || $_FILES['apk_file']['error'] !== UPLOAD_ERR_OK) {
                throw new Exception("Envie um arquivo .apk válido (modo LOCAL).");
            }

            $tmp  = $_FILES['apk_file']['tmp_name'];
            $name = $_FILES['apk_file']['name'] ?? 'app.apk';

            // valida extensão
            $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
            if ($ext !== 'apk') {
                throw new Exception("O arquivo enviado precisa ser .apk");
            }

            // nome seguro
            $safeBase = preg_replace('/[^a-zA-Z0-9_\-\.]/', '_', pathinfo($name, PATHINFO_FILENAME));
            if (!$safeBase) $safeBase = 'app';

            $finalName = $safeBase . '_v' . $version_code . '_' . time() . '.apk';
            $dest = $apksDir . '/' . $finalName;

            if (!@move_uploaded_file($tmp, $dest)) {
                throw new Exception("Falha ao salvar o APK em /api/apks. Verifique permissões.");
            }

            $local_file = $finalName;

            // gera URL automática para o apk local
            $baseDir = project_base_dir_from_this_script();
            $apk_url = base_url_for_dir($baseDir) . '/api/apks/' . rawurlencode($finalName);
        }

        $stmt = $pdo->prepare("
            UPDATE update_settings
            SET
                version_code = :vc,
                version_name = :vn,
                apk_source   = :src,
                apk_url      = :url,
                local_file   = :lf,
                changelog    = :cl,
                force        = :fo,
                updated_at   = :ua
            WHERE id = 1
        ");
        $stmt->execute([
            ':vc' => $version_code,
            ':vn' => $version_name,
            ':src' => $apk_source,
            ':url' => $apk_url,
            ':lf' => $local_file,
            ':cl' => $changelog,
            ':fo' => $force,
            ':ua' => date('Y-m-d H:i:s')
        ]);

        $message = "✅ Configuração salva com sucesso!";
    }

    // Carrega config atual
    $row = $pdo->query("SELECT * FROM update_settings WHERE id = 1")->fetch(PDO::FETCH_ASSOC);

} catch (Exception $e) {
    $error = $e->getMessage();
    $row = $row ?? [
        'version_code' => 0,
        'version_name' => '',
        'apk_source' => 'remote',
        'apk_url' => '',
        'local_file' => '',
        'changelog' => '',
        'force' => 0,
        'updated_at' => ''
    ];
}

$page_title = "📱 Painel de Atualização (APK)";

$page_content = '
<!-- Adicionando as bibliotecas necessárias no cabeçalho -->
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<div class="card">
    <div class="card-header">
        <h3 class="card-title">📱 Painel de Atualização (APK)</h3>
    </div>
    <div class="card-body">
        ' . ($message ? '<div class="alert alert-success alert-dismissible fade show" role="alert">✅ ' . h($message) . '<button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button></div>' : '') . '
        ' . ($error ? '<div class="alert alert-danger alert-dismissible fade show" role="alert">❌ ' . h($error) . '<button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button></div>' : '') . '

        <form method="post" enctype="multipart/form-data" id="updateForm">
            <div class="row mb-4">
                <div class="col-md-6">
                    <label class="form-label">🔢 version_code (obrigatório)</label>
                    <input type="number" class="form-control" name="version_code" min="1" value="' . h($row['version_code'] ?? 0) . '" required>
                    <small class="text-muted">Número inteiro que identifica a versão</small>
                </div>
                <div class="col-md-6">
                    <label class="form-label">🏷️ version_name</label>
                    <input type="text" class="form-control" name="version_name" value="' . h($row['version_name'] ?? '') . '" placeholder="Ex: 1.2.0">
                    <small class="text-muted">Nome legível da versão (ex: 1.2.0)</small>
                </div>
            </div>

            <div class="mb-4">
                <div class="form-check form-switch">
                    <input class="form-check-input" type="checkbox" name="force" id="forceCheck" ' . (((int)($row['force'] ?? 0) === 1) ? 'checked' : '') . '>
                    <label class="form-check-label" for="forceCheck">
                        ⚡ Forçar atualização obrigatória
                    </label>
                </div>
                <small class="text-muted d-block mt-2">Se ativado, o app não poderá ignorar a atualização</small>
            </div>

            <div class="mb-4">
                <label class="form-label">📝 Changelog (Notas de Atualização)</label>
                <textarea class="form-control" name="changelog" rows="5" placeholder="- Correções de bugs&#10;- Novas funcionalidades&#10;- Melhorias de performance">' . h($row['changelog'] ?? '') . '</textarea>
                <small class="text-muted">Descreva as mudanças desta versão</small>
            </div>

            <hr class="my-4">

            <div class="mb-4">
                <label class="form-label">📦 Fonte do APK</label>
                <select class="form-select" name="apk_source" id="apk_source" onchange="toggleSource()">
                    <option value="remote">🌐 URL Remota (link externo)</option>
                    <option value="local">📤 Upload Local (.apk)</option>
                </select>
                <small class="text-muted d-block mt-2">Escolha como o APK será distribuído</small>
            </select>
            </div>

            <div id="remoteBox" class="mb-4">
                <label class="form-label">🔗 URL do APK (remoto)</label>
                <input type="url" class="form-control" name="apk_url" value="' . h($row['apk_url'] ?? '') . '" placeholder="https://seusite.com/app.apk">
                <small class="text-muted">Cole o link direto para o arquivo APK</small>
            </div>

            <div id="localBox" class="mb-4" style="display:none;">
                <label class="form-label">📲 Enviar APK (.apk)</label>
                <div class="upload-box">
                    <input type="file" class="form-control" name="apk_file" id="apkFile" accept=".apk">
                    <div id="fileInfo" class="file-info" style="display:none; margin-top:10px;">
                        <small class="text-success">✅ Arquivo selecionado: <span id="fileName"></span></small>
                    </div>
                </div>
                <small class="text-muted d-block mt-2">
                    O APK será salvo em <b>/api/apks/</b> e a URL será gerada automaticamente
                </small>
            </div>

            <hr class="my-4">

            <div class="d-grid gap-2">
                <button class="btn btn-primary btn-lg" type="submit" id="submitBtn">
                    <span id="btnText">💾 Salvar Configurações</span>
                    <span id="btnLoader" style="display:none;">
                        <span class="spinner-border spinner-border-sm me-2" role="status" aria-hidden="true"></span>
                        Enviando...
                    </span>
                </button>
            </div>
        </form>

        <hr class="my-4">

        <div class="info-box">
            <h5>ℹ️ Configuração Atual</h5>
            <div class="info-grid">
                <div class="info-item">
                    <strong>📦 Fonte:</strong> <span class="badge bg-info">' . h($row['apk_source'] ?? 'remote') . '</span>
                </div>
                <div class="info-item">
                    <strong>🔗 URL:</strong> <code>' . (h($row['apk_url'] ?? '')) . '</code>
                </div>
                <div class="info-item">
                    <strong>📁 Arquivo Local:</strong> ' . (h($row['local_file'] ?? '') ? '<code>' . h($row['local_file'] ?? '') . '</code>' : '<span class="text-muted">Nenhum</span>') . '
                </div>
                <div class="info-item">
                    <strong>⏰ Atualizado em:</strong> ' . (h($row['updated_at'] ?? '') ? h($row['updated_at'] ?? '') : '<span class="text-muted">Nunca</span>') . '
                </div>
            </div>
        </div>
    </div>
</div>

<style>
    .form-label {
        font-weight: 600;
        margin-bottom: 0.5rem;
        color: #2c3e50;
    }
    
    .form-control, .form-select {
        border-radius: 0.5rem;
        border: 1px solid #dee2e6;
        transition: all 0.3s ease;
    }
    
    .form-control:focus, .form-select:focus {
        border-color: #0d6efd;
        box-shadow: 0 0 0 0.25rem rgba(13, 110, 253, 0.25);
    }
    
    .alert {
        border-radius: 0.5rem;
        border: none;
        font-weight: 500;
    }
    
    .btn-lg {
        font-weight: 600;
        font-size: 1rem;
        padding: 0.75rem 1.5rem;
        border-radius: 0.5rem;
        transition: all 0.3s ease;
    }
    
    .btn-primary:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(13, 110, 253, 0.3);
    }
    
    .upload-box {
        border: 2px dashed #0d6efd;
        border-radius: 0.5rem;
        padding: 1rem;
        background-color: #f8f9ff;
        transition: all 0.3s ease;
    }
    
    .upload-box:hover {
        background-color: #f0f4ff;
    }
    
    .file-info {
        padding: 0.75rem;
        background-color: #d1e7dd;
        border-radius: 0.375rem;
        border-left: 4px solid #198754;
    }
    
    .info-box {
        background-color: #e9ecef;
        border-left: 4px solid #0d6efd;
        padding: 1.5rem;
        border-radius: 0.5rem;
        margin-top: 1.5rem;
    }
    
    .info-box h5 {
        margin-bottom: 1rem;
        color: #000000;
        font-weight: 600;
    }
    
    .info-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 1rem;
    }
    
    .info-item {
        padding: 0.75rem;
        background-color: #d3d7dd;
        border-radius: 0.375rem;
        border: 1px solid #adb5bd;
        color: #000000;
        font-weight: 500;
    }
    
    .info-item code {
        background-color: #f8f9fa;
        padding: 0.25rem 0.5rem;
        border-radius: 0.25rem;
        color: #e83e8c;
        font-size: 0.875rem;
        word-break: break-all;
    }
    
    .badge {
        font-weight: 500;
        padding: 0.5rem 0.75rem;
    }
    
    .spinner-border-sm {
        width: 1rem;
        height: 1rem;
    }
    
    .form-check-input {
        width: 2rem;
        height: 2rem;
        cursor: pointer;
    }
    
    .form-check-label {
        font-weight: 500;
        cursor: pointer;
        font-size: 1rem;
    }
</style>

<script>
// Verificação de dependências
if (typeof $ === "undefined") {
    console.error("jQuery não está carregado!");
}

$(document).ready(function() {
    console.log("📱 Painel de Atualização carregado");
    
    function toggleSource() {
        const v = document.getElementById("apk_source").value;
        document.getElementById("remoteBox").style.display = (v === "remote") ? "block" : "none";
        document.getElementById("localBox").style.display = (v === "local") ? "block" : "none";
    }
    
    // Expõe a função globalmente
    window.toggleSource = toggleSource;
    
    // Chama na inicialização
    toggleSource();
    
    // Monitorar seleção de arquivo
    $("#apkFile").on("change", function() {
        const file = this.files[0];
        if (file) {
            const sizeMB = (file.size / (1024 * 1024)).toFixed(2);
            $("#fileName").text(file.name + " (" + sizeMB + " MB)");
            $("#fileInfo").show();
        } else {
            $("#fileInfo").hide();
        }
    });
    
    $("#updateForm").on("submit", function(e) {
        e.preventDefault();
        
        // Validações
        const versionCode = $("input[name=\"version_code\"]").val();
        const apkSource = $("#apk_source").val();
        
        if (!versionCode || versionCode <= 0) {
            Swal.fire({
                title: "❌ Erro de Validação",
                text: "Por favor, insira um version_code válido (maior que 0)",
                icon: "error"
            });
            return;
        }
        
        if (apkSource === "remote") {
            const apkUrl = $("input[name=\"apk_url\"]").val();
            if (!apkUrl) {
                Swal.fire({
                    title: "❌ Erro de Validação",
                    text: "Por favor, insira uma URL válida para o APK remoto",
                    icon: "error"
                });
                return;
            }
        } else if (apkSource === "local") {
            const apkFile = $("#apkFile")[0].files[0];
            if (!apkFile) {
                Swal.fire({
                    title: "❌ Erro de Validação",
                    text: "Por favor, selecione um arquivo APK para upload",
                    icon: "error"
                });
                return;
            }
            
            if (apkFile.size > 500 * 1024 * 1024) { // 500MB
                Swal.fire({
                    title: "❌ Arquivo muito grande",
                    text: "O arquivo APK não pode exceder 500 MB",
                    icon: "error"
                });
                return;
            }
        }
        
        // Mostrar loader
        $("#submitBtn").prop("disabled", true);
        $("#btnText").hide();
        $("#btnLoader").show();
        
        const formData = new FormData(this);
        
        $.ajax({
            url: window.location.href,
            method: "POST",
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                console.log("✅ Resposta recebida");
                
                // Esconde loader
                $("#submitBtn").prop("disabled", false);
                $("#btnText").show();
                $("#btnLoader").hide();
                
                Swal.fire({
                    title: "✅ Sucesso!",
                    text: "Configuração salva com sucesso! A página será recarregada...",
                    icon: "success",
                    timer: 2000,
                    timerProgressBar: true
                }).then(() => {
                    location.reload();
                });
            },
            error: function(xhr, status, error) {
                console.error("❌ Erro na requisição:", status, error);
                
                // Esconde loader
                $("#submitBtn").prop("disabled", false);
                $("#btnText").show();
                $("#btnLoader").hide();
                
                Swal.fire({
                    title: "❌ Erro",
                    text: "Ocorreu um erro ao salvar a configuração. Tente novamente.",
                    icon: "error"
                });
            }
        });
    });
});
</script>
';

include 'includes/layout.php';
?>
