<?php
ini_set("display_errors", 1);
error_reporting(E_ALL);

include "includes/header.php";

$adb = new SQLite3('./api/studiolivecode.db');

// Criação das tabelas com a estrutura correta
$adb->exec("CREATE TABLE IF NOT EXISTS codes (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    dns_id INTEGER NOT NULL,
    code TEXT NOT NULL,
    m3u_url TEXT NOT NULL,
    mac_address TEXT,
    cliente TEXT,
    FOREIGN KEY (dns_id) REFERENCES dns(id)
)");

$adb->exec("CREATE TABLE IF NOT EXISTS dns (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    title TEXT NOT NULL,
    url TEXT NOT NULL
)");

$adb->exec("CREATE TABLE IF NOT EXISTS playlist (
    id INTEGER PRIMARY KEY,
    dns_id INTEGER,
    mac_address TEXT NOT NULL,
    username TEXT NOT NULL,
    password TEXT NOT NULL,
    pin TEXT NOT NULL,
    link TEXT,
    type VARCHAR(50),
    url VARCHAR(50)
)");

// Função para manipular o arquivo JSON
function manipularCodigoJSON($codigo, $acao) {
    $jsonFile = 'codigos_ativacao.json';
    
    // Carrega os dados existentes ou cria um array vazio
    $currentData = file_exists($jsonFile) ? json_decode(file_get_contents($jsonFile), true) : [];
    
    if ($acao === 'adicionar') {
        // Adiciona o código se não existir
        if (!in_array($codigo, $currentData)) {
            $currentData[] = $codigo;
            file_put_contents($jsonFile, json_encode($currentData, JSON_PRETTY_PRINT));
        }
    } elseif ($acao === 'remover') {
        // Remove o código se existir
        $key = array_search($codigo, $currentData);
        if ($key !== false) {
            unset($currentData[$key]);
            // Reindexa o array e salva
            file_put_contents($jsonFile, json_encode(array_values($currentData), JSON_PRETTY_PRINT));
        }
    }
}

// Processamento dos formulários
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'add':
                $codigo = trim($_POST['codigo'] ?? '');
                $m3u_url = trim($_POST['m3u_url'] ?? '');
                $dns_id = $_POST['dns_id'] ?? null;
                $mac_address = trim($_POST['mac_address'] ?? '');
                $cliente = trim($_POST['cliente'] ?? '');

                // Validação dos campos obrigatórios
                if (empty($cliente)) {
                    echo "<div class='alert alert-danger'>⚠️ O campo Nome Cliente é obrigatório.</div>";
                    break;
                }

                // Validação do formato MAC Address
                if (!empty($mac_address) && !preg_match('/^([0-9A-Fa-f]{2}[:-]){5}([0-9A-Fa-f]{2})$/', $mac_address)) {
                    echo "<div class='alert alert-danger'>⚠️ Formato de MAC Address inválido. Use o formato: XX:XX:XX:XX:XX:XX</div>";
                    break;
                }

                if (empty($codigo) || empty($m3u_url) || empty($dns_id)) {
                    echo "<div class='alert alert-danger'>⚠️ Preencha todos os campos obrigatórios.</div>";
                } else {
                    // Verifica se já existe um código IDÊNTICO para este DNS
                    $stmt = $adb->prepare("SELECT COUNT(*) FROM codes WHERE dns_id = :dns_id AND code = :code");
                    $stmt->bindValue(':dns_id', $dns_id, SQLITE3_INTEGER);
                    $stmt->bindValue(':code', $codigo, SQLITE3_TEXT);
                    $exists = $stmt->execute()->fetchArray()[0];

                    if ($exists) {
                        echo "<div class='alert alert-warning'>⚠️ Já existe um código IDÊNTICO cadastrado para esta Conexão DNS.</div>";
                    } else {
                        $stmt = $adb->prepare("INSERT INTO codes (dns_id, code, m3u_url, mac_address, cliente) VALUES (:dns_id, :code, :m3u_url, :mac_address, :cliente)");
                        $stmt->bindValue(':dns_id', $dns_id, SQLITE3_INTEGER);
                        $stmt->bindValue(':code', $codigo, SQLITE3_TEXT);
                        $stmt->bindValue(':m3u_url', $m3u_url, SQLITE3_TEXT);
                        $stmt->bindValue(':mac_address', $mac_address, SQLITE3_TEXT);
                        $stmt->bindValue(':cliente', $cliente, SQLITE3_TEXT);

                        if ($stmt->execute()) {
                            // Adiciona o código ao JSON
                            manipularCodigoJSON($codigo, 'adicionar');
                            
                            echo "<div class='alert alert-success'>✅ Código cadastrado com sucesso!</div>";
                            
                            // Inserir na tabela playlist se MAC Address foi fornecido
                            if (!empty($mac_address)) {
                                $parsed_url = parse_url($m3u_url);
                                parse_str($parsed_url['query'], $query_params);
                                
                                $username = $query_params['username'] ?? 'default';
                                $password = $query_params['password'] ?? 'default';
                                
                                $stmt_playlist = $adb->prepare("INSERT INTO playlist (dns_id, mac_address, username, password, pin, link, type, url) 
                                    VALUES (:dns_id, :mac_address, :username, :password, :pin, :link, :type, :url)");
                                
                                $stmt_playlist->bindValue(':dns_id', $dns_id, SQLITE3_INTEGER);
                                $stmt_playlist->bindValue(':mac_address', $mac_address, SQLITE3_TEXT);
                                $stmt_playlist->bindValue(':username', $username, SQLITE3_TEXT);
                                $stmt_playlist->bindValue(':password', $password, SQLITE3_TEXT);
                                $stmt_playlist->bindValue(':pin', '0000', SQLITE3_TEXT);
                                $stmt_playlist->bindValue(':link', '', SQLITE3_TEXT);
                                $stmt_playlist->bindValue(':type', 'm3u', SQLITE3_TEXT);
                                $stmt_playlist->bindValue(':url', $m3u_url, SQLITE3_TEXT);
                                
                                if ($stmt_playlist->execute()) {
                                    echo "<div class='alert alert-info'>📡 Dispositivo com MAC {$mac_address} adicionado à playlist<br>
                                          👤 Username: {$username}<br>
                                          🔑 Password: {$password}<br>
                                          👥 Cliente: {$cliente}</div>";
                                }
                            }
                        } else {
                            echo "<div class='alert alert-danger'>⚠️ Erro ao cadastrar código.</div>";
                        }
                    }
                }
                break;

            case 'edit':
                $id = $_POST['id'] ?? null;
                $new_m3u_url = trim($_POST['new_m3u_url'] ?? '');
                $new_mac_address = trim($_POST['new_mac_address'] ?? '');
                $new_cliente = trim($_POST['new_cliente'] ?? '');

                if (empty($new_cliente)) {
                    echo "<div class='alert alert-danger'>⚠️ O campo Nome Cliente é obrigatório.</div>";
                    break;
                }

                if (empty($id) || empty($new_m3u_url)) {
                    echo "<div class='alert alert-danger'>⚠️ Dados inválidos para edição.</div>";
                } else {
                    $stmt = $adb->prepare("UPDATE codes SET m3u_url = :m3u_url, mac_address = :mac_address, cliente = :cliente WHERE id = :id");
                    $stmt->bindValue(':id', $id, SQLITE3_INTEGER);
                    $stmt->bindValue(':m3u_url', $new_m3u_url, SQLITE3_TEXT);
                    $stmt->bindValue(':mac_address', $new_mac_address, SQLITE3_TEXT);
                    $stmt->bindValue(':cliente', $new_cliente, SQLITE3_TEXT);

                    if ($stmt->execute()) {
                        echo "<div class='alert alert-success'>✅ URL M3U e MAC Address atualizados com sucesso!</div>";
                        
                        if (!empty($new_mac_address)) {
                            $parsed_url = parse_url($new_m3u_url);
                            parse_str($parsed_url['query'], $query_params);
                            
                            $new_username = $query_params['username'] ?? 'default';
                            $new_password = $query_params['password'] ?? 'default';
                            
                            $stmt_playlist = $adb->prepare("UPDATE playlist SET 
                                url = :url, 
                                username = :username,
                                password = :password
                                WHERE mac_address = :mac_address");
                                
                            $stmt_playlist->bindValue(':url', $new_m3u_url, SQLITE3_TEXT);
                            $stmt_playlist->bindValue(':username', $new_username, SQLITE3_TEXT);
                            $stmt_playlist->bindValue(':password', $new_password, SQLITE3_TEXT);
                            $stmt_playlist->bindValue(':mac_address', $new_mac_address, SQLITE3_TEXT);
                            
                            $stmt_playlist->execute();
                            
                            echo "<div class='alert alert-info'>📡 Playlist atualizada com novos dados:<br>
                                  👤 Username: {$new_username}<br>
                                  🔑 Password: {$new_password}<br>
                                  👥 Cliente: {$new_cliente}</div>";
                        }
                    } else {
                        echo "<div class='alert alert-danger'>⚠️ Erro ao atualizar URL M3U e MAC Address.</div>";
                    }
                }
                break;

            case 'delete':
                $id = $_POST['id'] ?? null;
                $mac_address = $_POST['mac_address'] ?? '';
                
                if (empty($id)) {
                    echo "<div class='alert alert-danger'>⚠️ ID inválido para exclusão.</div>";
                } else {
                    // Obtém o código antes de deletar
                    $stmt = $adb->prepare("SELECT code FROM codes WHERE id = :id");
                    $stmt->bindValue(':id', $id, SQLITE3_INTEGER);
                    $result = $stmt->execute();
                    $row = $result->fetchArray(SQLITE3_ASSOC);
                    $codigo = $row['code'] ?? '';
                    
                    // Deleta da tabela codes
                    $stmt = $adb->prepare("DELETE FROM codes WHERE id = :id");
                    $stmt->bindValue(':id', $id, SQLITE3_INTEGER);
                    
                    if ($stmt->execute()) {
                        // Remove do arquivo JSON
                        if (!empty($codigo)) {
                            manipularCodigoJSON($codigo, 'remover');
                        }
                        
                        echo "<div class='alert alert-success'>🗑️ Código deletado com sucesso!</div>";
                        
                        // Deleta da playlist se tiver MAC Address
                        if (!empty($mac_address)) {
                            $stmt_playlist = $adb->prepare("DELETE FROM playlist WHERE mac_address = :mac_address");
                            $stmt_playlist->bindValue(':mac_address', $mac_address, SQLITE3_TEXT);
                            $stmt_playlist->execute();
                            echo "<div class='alert alert-info'>📡 Dispositivo com MAC {$mac_address} removido da playlist</div>";
                        }
                    } else {
                        echo "<div class='alert alert-danger'>⚠️ Erro ao deletar código.</div>";
                    }
                }
                break;
        }
    }
}

// Buscar conexões DNS
$dns_connections = [];
$result = $adb->query("SELECT id, title FROM dns ORDER BY title ASC");
while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
    $dns_connections[] = $row;
}

// Processar busca se existir
$search_term = '';
$codigos = [];
if (isset($_GET['search']) && !empty($_GET['search'])) {
    $search_term = trim($_GET['search']);
    $stmt = $adb->prepare("
        SELECT c.id, c.dns_id, c.code, c.m3u_url, c.mac_address, c.cliente, d.title AS dns_title 
        FROM codes c 
        JOIN dns d ON c.dns_id = d.id 
        WHERE c.code LIKE :search OR c.mac_address LIKE :search OR c.cliente LIKE :search
        ORDER BY c.id DESC
    ");
    $stmt->bindValue(':search', "%$search_term%", SQLITE3_TEXT);
    $result = $stmt->execute();
} else {
    $result = $adb->query("
        SELECT c.id, c.dns_id, c.code, c.m3u_url, c.mac_address, c.cliente, d.title AS dns_title 
        FROM codes c 
        JOIN dns d ON c.dns_id = d.id 
        ORDER BY c.id DESC
    ");
}

while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
    $codigos[] = $row;
}
?>

<style>
    .texto-amarelo {
        color: yellow !important;
    }
    .search-box {
        margin-bottom: 20px;
    }
</style>

<div class="container">
    <h2 class="text-center my-4">📝 Cadastro de Código M3U</h2>
    
    <!-- Barra de busca -->
    <div class="card mb-4">
        <div class="card-header bg-dark text-white">
            <h4 class="mb-0">🔍 Buscar Códigos</h4>
        </div>
        <div class="card-body">
            <form method="get" class="search-box">
                <div class="input-group">
                    <input type="text" class="form-control" name="search" placeholder="Digite código, MAC address ou cliente" value="<?= htmlspecialchars($search_term) ?>">
                    <div class="input-group-append">
                        <button class="btn btn-danger" type="submit">Buscar</button>
                        <?php if (!empty($search_term)): ?>
                            <a href="?" class="btn btn-secondary">Limpar</a>
                        <?php endif; ?>
                    </div>
                </div>
                <small class="form-text text-muted texto-amarelo">
                    Você pode buscar por código de ativação, endereço MAC ou nome do cliente
                </small>
            </form>
        </div>
    </div>
    
    <div class="card mb-4">
        <div class="card-header bg-dark text-white">
            <h4 class="mb-0">➕ Adicionar Novo Código</h4>
        </div>
        <div class="card-body">
            <form method="post">
                <input type="hidden" name="action" value="add">
                
                <div class="form-group">
                    <label>🔑 Código:</label>
                    <input type="text" class="form-control" name="codigo" required>
                </div>
                
                <div class="form-group">
                    <label>🔗 URL M3U:</label>
                    <textarea class="form-control" name="m3u_url" rows="3" required placeholder="http://exemplo.com/get.php?username=123456&password=654321&type=m3u&output=m3u8"></textarea>
                    <small class="form-text text-muted texto-amarelo">
                        Exemplo: http://exemplo.com/get.php?username=123456&password=654321&type=m3u&output=m3u8
                    </small>
                </div>
                
                <div class="form-group">
                    <label>📡 MAC Address (opcional):</label>
                    <input type="text" class="form-control" name="mac_address" placeholder="Ex: E6:40:E7:89:DF:3B" pattern="^([0-9A-Fa-f]{2}[:-]){5}([0-9A-Fa-f]{2})$">
                    <small class="form-text text-muted texto-amarelo">
                        Atenção: O MAC Address é preenchido automaticamente pelo App quando o cliente digita o código.
                    </small>
                </div>
                
                <div class="form-group">
                    <label>👤 Nome Cliente:</label>
                    <input type="text" class="form-control" name="cliente" placeholder="Nome do cliente" required>
                </div>
                
                <div class="form-group">
                    <label>🌐 Conexão DNS:</label>
                    <select class="form-control" name="dns_id" required>
                        <option value="">Selecione uma Conexão DNS</option>
                        <?php foreach ($dns_connections as $dns): ?>
                            <option value="<?= htmlspecialchars($dns['id']) ?>">
                                <?= htmlspecialchars($dns['title']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <button type="submit" class="btn btn-danger btn-block mt-3">💾 Cadastrar Código</button>
            </form>
        </div>
    </div>

    <?php if (!empty($codigos)): ?>
        <div class="card">
            <div class="card-header bg-dark text-white">
                <h4 class="mb-0">📋 Códigos Cadastrados (<?= count($codigos) ?>)</h4>
            </div>
            <div class="card-body">
                <?php if (!empty($search_term)): ?>
                    <div class="alert alert-info mb-3">
                        Resultados da busca por: <strong><?= htmlspecialchars($search_term) ?></strong>
                    </div>
                <?php endif; ?>
                
                <div class="row">
                    <?php foreach ($codigos as $c): ?>
                        <div class="col-md-6 col-lg-4 mb-4">
                            <div class="card h-100 border-dark">
                                <div class="card-header bg-dark text-white">
                                    <h5 class="card-title mb-0">
                                        🔑 <?= htmlspecialchars($c['code']) ?>
                                    </h5>
                                </div>
                                <div class="card-body">
                                    <div class="mb-2">
                                        <strong>#️⃣ ID:</strong> <?= htmlspecialchars($c['id']) ?>
                                    </div>
                                    <div class="mb-2">
                                        <strong>🌐 DNS:</strong> <?= htmlspecialchars($c['dns_title']) ?> (ID: <?= htmlspecialchars($c['dns_id']) ?>)
                                    </div>
                                    <div class="mb-2">
                                        <strong>📡 MAC:</strong> <?= !empty($c['mac_address']) ? htmlspecialchars($c['mac_address']) : 'N/A' ?>
                                    </div>
                                    <?php if (!empty($c['cliente'])): ?>
                                    <div class="mb-2">
                                        <strong>👤 Cliente:</strong> <?= htmlspecialchars($c['cliente']) ?>
                                    </div>
                                    <?php endif; ?>
                                    <div class="mb-3">
                                        <strong>🔗 M3U:</strong>
                                        <button class="btn btn-sm btn-info btn-block mt-1" onclick="copyToClipboard('<?= htmlspecialchars($c['m3u_url']) ?>')">
                                            📋 Copiar URL
                                        </button>
                                    </div>
                                </div>
                                <div class="card-footer bg-light">
                                    <div class="btn-group w-100">
                                        <button class="btn btn-sm btn-warning" data-toggle="modal" data-target="#editModal<?= $c['id'] ?>">
                                            ✏️ Editar
                                        </button>
                                        <button class="btn btn-sm btn-danger" data-toggle="modal" data-target="#deleteModal<?= $c['id'] ?>">
                                            🗑️ Deletar
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Modal de Edição -->
                        <div class="modal fade" id="editModal<?= $c['id'] ?>" tabindex="-1" role="dialog" aria-hidden="true">
                            <div class="modal-dialog">
                                <div class="modal-content bg-dark">
                                    <div class="modal-header">
                                        <h5 class="modal-title">✏️ Editar URL M3U e MAC Address</h5>
                                        <button type="button" class="close text-white" data-dismiss="modal">&times;</button>
                                    </div>
                                    <form method="post">
                                        <div class="modal-body">
                                            <input type="hidden" name="action" value="edit">
                                            <input type="hidden" name="id" value="<?= $c['id'] ?>">
                                            
                                            <div class="form-group">
                                                <label>🔗 Nova URL M3U:</label>
                                                <textarea class="form-control" name="new_m3u_url" rows="5" required><?= htmlspecialchars($c['m3u_url']) ?></textarea>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label>📡 Novo MAC Address:</label>
                                                <input type="text" class="form-control" name="new_mac_address" 
                                                       value="<?= htmlspecialchars($c['mac_address'] ?? '') ?>" 
                                                       placeholder="Ex: E6:40:E7:89:DF:3B" 
                                                       pattern="^([0-9A-Fa-f]{2}[:-]){5}([0-9A-Fa-f]{2})$">
                                                <small class="form-text text-muted">Deixe em branco para remover</small>
                                            </div>
                                            
                                            <div class="form-group">
                                                <label>👤 Nome Cliente:</label>
                                                <input type="text" class="form-control" name="new_cliente" 
                                                       value="<?= htmlspecialchars($c['cliente'] ?? '') ?>" 
                                                       placeholder="Nome do cliente" required>
                                            </div>
                                        </div>
                                        <div class="modal-footer">
                                            <button type="button" class="btn btn-secondary" data-dismiss="modal">❌ Cancelar</button>
                                            <button type="submit" class="btn btn-primary">💾 Salvar</button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Modal de Exclusão -->
                        <div class="modal fade" id="deleteModal<?= $c['id'] ?>" tabindex="-1" role="dialog" aria-hidden="true">
                            <div class="modal-dialog">
                                <div class="modal-content bg-dark">
                                    <div class="modal-header">
                                        <h5 class="modal-title">🗑️ Confirmar Exclusão</h5>
                                        <button type="button" class="close text-white" data-dismiss="modal">&times;</button>
                                    </div>
                                    <form method="post">
                                        <div class="modal-body">
                                            <p>Tem certeza que deseja deletar o código <strong><?= htmlspecialchars($c['code']) ?></strong>?</p>
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="id" value="<?= $c['id'] ?>">
                                            <input type="hidden" name="mac_address" value="<?= htmlspecialchars($c['mac_address'] ?? '') ?>">
                                        </div>
                                        <div class="modal-footer">
                                            <button type="button" class="btn btn-secondary" data-dismiss="modal">❌ Cancelar</button>
                                            <button type="submit" class="btn btn-danger">🗑️ Confirmar</button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    <?php else: ?>
        <div class="alert alert-warning">
            <?php if (!empty($search_term)): ?>
                Nenhum código encontrado para a busca: <strong><?= htmlspecialchars($search_term) ?></strong>
            <?php else: ?>
                Nenhum código cadastrado ainda.
            <?php endif; ?>
        </div>
    <?php endif; ?>
</div>

<script>
function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(function() {
        alert('URL copiada para a área de transferência!');
    }, function(err) {
        console.error('Erro ao copiar texto: ', err);
    });
}
</script>

<?php include "includes/footer.php"; ?>