<?php
session_start();
ini_set("display_errors", 1);
error_reporting(E_ALL);

include "includes/header.php";

$adb = new SQLite3('./api/studiolivecode.db');

// Processamento dos formulários (apenas delete e edit agora)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'edit':
                handleEditCode($adb);
                break;
            case 'delete':
                handleDeleteCode($adb);
                break;
        }
    }
}

// Funções de processamento
function handleEditCode($adb) {
    if (empty($_POST['id']) || empty($_POST['m3u_url']) || empty($_POST['cliente'])) {
        echo alert('danger', "⚠️ Dados inválidos para edição.");
        return;
    }

    if (!empty($_POST['mac_address']) && !preg_match('/^([0-9A-Fa-f]{2}[:-]){5}([0-9A-Fa-f]{2})$/', $_POST['mac_address'])) {
        echo alert('danger', "⚠️ Formato de MAC Address inválido.");
        return;
    }

    $stmt = $adb->prepare("UPDATE codes SET 
                          m3u_url = :m3u_url, 
                          mac_address = :mac_address,
                          cliente = :cliente
                          WHERE id = :id");
    
    bindValues($stmt, [
        'id' => $_POST['id'],
        'm3u_url' => $_POST['m3u_url'],
        'mac_address' => $_POST['mac_address'] ?? '',
        'cliente' => $_POST['cliente']
    ]);

    if ($stmt->execute()) {
        echo alert('success', "✅ Código atualizado com sucesso!");
    } else {
        echo alert('danger', "⚠️ Erro ao atualizar código.");
    }
}

function handleDeleteCode($adb) {
    if (empty($_POST['id'])) {
        echo alert('danger', "⚠️ ID inválido para exclusão.");
        return;
    }

    // Busca o mac_address do código antes de deletar
    $stmtBusca = $adb->prepare("SELECT mac_address FROM codes WHERE id = :id");
    $stmtBusca->bindValue(':id', $_POST['id'], SQLITE3_INTEGER);
    $result = $stmtBusca->execute();
    $row = $result->fetchArray(SQLITE3_ASSOC);
    $mac_address = trim($row['mac_address'] ?? '');

    if (empty($mac_address)) {
        echo alert('danger', "⚠️ MAC Address não encontrado para exclusão na playlist.");
        return;
    }

    $adb->exec('BEGIN TRANSACTION');

    try {
        // Deleta registros na playlist onde o mac_address bate
        $stmtPlaylist = $adb->prepare("DELETE FROM playlist WHERE TRIM(mac_address) = :mac_address");
        $stmtPlaylist->bindValue(':mac_address', $mac_address, SQLITE3_TEXT);
        $stmtPlaylist->execute();

        // Deleta o código na tabela codes
        $stmt = $adb->prepare("DELETE FROM codes WHERE id = :id");
        $stmt->bindValue(':id', $_POST['id'], SQLITE3_INTEGER);
        $stmt->execute();

        $adb->exec('COMMIT');

        echo alert('success', "🗑️ Código e cadastros vinculados na playlist deletados com sucesso!");
    } catch (Exception $e) {
        $adb->exec('ROLLBACK');
        echo alert('danger', "⚠️ Erro ao deletar: " . $e->getMessage());
    }
}

// Funções auxiliares
function alert($type, $message) {
    return "<div class='alert alert-$type'>$message</div>";
}

function bindValues($stmt, $values) {
    foreach ($values as $key => $value) {
        $stmt->bindValue(":$key", $value, is_int($value) ? SQLITE3_INTEGER : SQLITE3_TEXT);
    }
}

// Busca os códigos cadastrados
function getCodes($adb, $search = '') {
    $query = "SELECT c.id, c.code, c.m3u_url, c.mac_address, c.cliente, 
                     d.id as dns_id, d.title as dns_title 
              FROM codes c 
              JOIN dns d ON c.dns_id = d.id";
    
    if (!empty($search)) {
        $query .= " WHERE c.code LIKE :search OR c.cliente LIKE :search OR c.mac_address LIKE :search";
    }
    
    $query .= " ORDER BY c.id DESC";
    
    $stmt = $adb->prepare($query);
    
    if (!empty($search)) {
        $stmt->bindValue(':search', "%$search%", SQLITE3_TEXT);
    }
    
    $result = $stmt->execute();
    $codes = [];
    
    while ($row = $result->fetchArray(SQLITE3_ASSOC)) {
        $codes[] = $row;
    }
    
    return $codes;
}

// Processa busca
$search_term = $_GET['search'] ?? '';
$codigos = getCodes($adb, $search_term);
?>

<div class="container">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2 class="mb-0">📋 Gerenciador de Códigos</h2>
        <a href="tutorial.php" class="btn btn-primary">➕ Cadastro Rápido</a>
    </div>

    <!-- Barra de busca -->
    <div class="card mb-4">
        <div class="card-header bg-dark text-white">
            <h4 class="mb-0">🔍 Buscar Códigos</h4>
        </div>
        <div class="card-body">
            <form method="get" class="search-box">
                <div class="input-group">
                    <input type="text" class="form-control" name="search" 
                           placeholder="Digite código, cliente ou MAC" 
                           value="<?= htmlspecialchars($search_term) ?>">
                    <div class="input-group-append">
                        <button class="btn btn-danger" type="submit">Buscar</button>
                        <?php if (!empty($search_term)): ?>
                            <a href="gerenciador.php" class="btn btn-secondary">Limpar</a>
                        <?php endif; ?>
                    </div>
                </div>
                <small class="form-text text-muted">
                    Você pode buscar por código de ativação, nome do cliente ou endereço MAC
                </small>
            </form>
        </div>
    </div>

    <!-- Lista de códigos -->
    <div class="card">
        <div class="card-header bg-dark text-white">
            <h4 class="mb-0">📋 Códigos Cadastrados (<?= count($codigos) ?>)</h4>
        </div>
        <div class="card-body">
            <?php if (!empty($search_term)): ?>
                <div class="alert alert-info mb-3">
                    Resultados da busca por: <strong><?= htmlspecialchars($search_term) ?></strong>
                </div>
            <?php endif; ?>
            
            <?php if (empty($codigos)): ?>
                <div class="alert alert-warning">
                    Nenhum código encontrado. <a href="tutorial.php">Cadastre um novo código</a>
                </div>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead class="thead-dark">
                            <tr>
                                <th>ID</th>
                                <th>Código</th>
                                <th>Cliente</th>
                                <th>DNS</th>
                                <th>MAC</th>
                                <th>Ações</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($codigos as $c): ?>
                                <tr>
                                    <td><?= $c['id'] ?></td>
                                    <td><strong><?= htmlspecialchars($c['code']) ?></strong></td>
                                    <td><?= htmlspecialchars($c['cliente']) ?></td>
                                    <td><?= htmlspecialchars($c['dns_title']) ?></td>
                                    <td><?= !empty($c['mac_address']) ? htmlspecialchars($c['mac_address']) : 'N/A' ?></td>
                                    <td>
                                        <button class="btn btn-sm btn-warning" data-toggle="modal" 
                                                data-target="#editModal<?= $c['id'] ?>">
                                            ✏️ Editar
                                        </button>
                                        <button class="btn btn-sm btn-danger" data-toggle="modal" 
                                                data-target="#deleteModal<?= $c['id'] ?>">
                                            🗑️ Excluir
                                        </button>
                                    </td>
                                </tr>
                                
                                <!-- Modal de Edição -->
                                <div class="modal fade" id="editModal<?= $c['id'] ?>" tabindex="-1" role="dialog">
                                    <div class="modal-dialog" role="document">
                                        <div class="modal-content">
                                            <div class="modal-header">
                                                <h5 class="modal-title">Editar Código <?= $c['code'] ?></h5>
                                                <button type="button" class="close" data-dismiss="modal">&times;</button>
                                            </div>
                                            <form method="post">
                                                <div class="modal-body">
                                                    <input type="hidden" name="action" value="edit">
                                                    <input type="hidden" name="id" value="<?= $c['id'] ?>">
                                                    
                                                    <div class="form-group">
                                                        <label>🔗 URL M3U:</label>
                                                        <textarea class="form-control" name="m3u_url" rows="3" required><?= 
                                                            htmlspecialchars($c['m3u_url']) ?></textarea>
                                                    </div>
                                                    
                                                    <div class="form-group">
                                                        <label>📡 MAC Address:</label>
                                                        <input type="text" class="form-control" name="mac_address"
                                                               value="<?= htmlspecialchars($c['mac_address']) ?>"
                                                               placeholder="XX:XX:XX:XX:XX:XX"
                                                               pattern="^([0-9A-Fa-f]{2}[:-]){5}([0-9A-Fa-f]{2})$">
                                                    </div>
                                                    
                                                    <div class="form-group">
                                                        <label>👤 Nome Cliente:</label>
                                                        <input type="text" class="form-control" name="cliente"
                                                               value="<?= htmlspecialchars($c['cliente']) ?>" required>
                                                    </div>
                                                </div>
                                                <div class="modal-footer">
                                                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancelar</button>
                                                    <button type="submit" class="btn btn-primary">Salvar</button>
                                                </div>
                                            </form>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Modal de Exclusão -->
                                <div class="modal fade" id="deleteModal<?= $c['id'] ?>" tabindex="-1" role="dialog">
                                    <div class="modal-dialog" role="document">
                                        <div class="modal-content">
                                            <div class="modal-header bg-danger text-white">
                                                <h5 class="modal-title">Confirmar Exclusão</h5>
                                                <button type="button" class="close text-white" data-dismiss="modal">&times;</button>
                                            </div>
                                            <form method="post">
                                                <div class="modal-body">
                                                    <p>Tem certeza que deseja excluir o código <strong><?= $c['code'] ?></strong>?</p>
                                                    <input type="hidden" name="action" value="delete">
                                                    <input type="hidden" name="id" value="<?= $c['id'] ?>">
                                                </div>
                                                <div class="modal-footer">
                                                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancelar</button>
                                                    <button type="submit" class="btn btn-danger">Confirmar Exclusão</button>
                                                </div>
                                            </form>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<?php include "includes/footer.php"; ?>
