<?php
declare(strict_types=1);
require_once __DIR__ . '/config.php';

/**
 * Ensures the user is authenticated before accessing a page.
 */
function require_login(): void {
    if (session_status() !== PHP_SESSION_ACTIVE) session_start();
    if (empty($_SESSION['panel_ok'])) {
        header('Location: login.php');
        exit;
    }
}

/**
 * Reads and normalizes the panel JSON data.
 */
function read_panel_data(): array {
    global $DATA_FILE, $DEFAULT_JSON;
    
    if (!file_exists($DATA_FILE)) {
        write_panel_data($DEFAULT_JSON);
    }
    
    $raw = @file_get_contents($DATA_FILE);
    $data = json_decode($raw ?: '', true);
    
    if (!is_array($data)) {
        $data = $DEFAULT_JSON;
    }

    // Normalize Core Structure
    if (!isset($data['dns']) || !is_array($data['dns'])) $data['dns'] = $DEFAULT_JSON['dns'];
    if (!isset($data['active_dns'])) $data['active_dns'] = 1;
    if (!isset($data['version'])) $data['version'] = '1.0';
    if (!isset($data['updated_at'])) $data['updated_at'] = gmdate('c');

    // Ensure DNS slots 1 through 5 are always present
    $byId = [];
    foreach ($data['dns'] as $row) {
        if (!isset($row['id'])) continue;
        $byId[(int)$row['id']] = $row;
    }
    
    $dns = [];
    for ($i = 1; $i <= 5; $i++) {
        $row = $byId[$i] ?? ['id' => $i, 'name' => 'DNS ' . $i, 'url' => '', 'enabled' => false];
        $row['id'] = $i;
        $row['name'] = (string)($row['name'] ?? ('DNS ' . $i));
        $row['url'] = trim((string)($row['url'] ?? ''));
        $row['enabled'] = (bool)($row['enabled'] ?? false);
        $dns[] = $row;
    }
    
    $data['dns'] = $dns;
    $data['active_dns'] = max(1, min(5, (int)$data['active_dns']));
    
    return $data;
}

/**
 * Atomic write to the JSON data file.
 */
function write_panel_data(array $data): void {
    global $DATA_FILE;
    
    $data['updated_at'] = gmdate('c');
    $json = json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    
    if ($json === false) {
        throw new RuntimeException('JSON encode failed');
    }
    
    $tmp = $DATA_FILE . '.tmp';
    
    // Using LOCK_EX for process safety
    if (@file_put_contents($tmp, $json, LOCK_EX) === false) {
        throw new RuntimeException('Write failed: Check folder permissions');
    }
    
    @rename($tmp, $DATA_FILE);
}

/**
 * Escapes HTML for safe output.
 */
function h(string $s): string { 
    return htmlspecialchars($s, ENT_QUOTES, 'UTF-8'); 
}

/**
 * Generates the base URL for the panel.
 */
function base_url(): string {
    $https = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') || 
             (($_SERVER['HTTP_X_FORWARDED_PROTO'] ?? '') === 'https');
             
    $scheme = $https ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $path = rtrim(str_replace('\\', '/', dirname($_SERVER['SCRIPT_NAME'] ?? '/')), '/');
    
    return $scheme . '://' . $host . ($path ? $path : '');
}