<?php
// panel/mdns/proxy.php
// Multi-DNS reverse proxy controlled by ../panel.json
// Point the app's base URL to: https://YOURDOMAIN/panel/mdns/  (note trailing slash)
// Requests like /player_api.php?... will be forwarded to the ACTIVE DNS.

// SECURITY NOTE: This is a transparent proxy. Protect your panel folder if needed.

declare(strict_types=1);

header('X-MDNS-Proxy: 1');

$panelJson = dirname(__DIR__) . '/panel.json';
if (!file_exists($panelJson)) {
    http_response_code(500);
    echo "panel.json missing";
    exit;
}

$cfg = json_decode((string)file_get_contents($panelJson), true);
if (!is_array($cfg)) {
    http_response_code(500);
    echo "panel.json invalid";
    exit;
}

$active = (int)($cfg['active_dns'] ?? 0);
$dnsList = $cfg['dns'] ?? [];
if (!is_array($dnsList) || empty($dnsList)) {
    http_response_code(500);
    echo "dns list missing";
    exit;
}

function pick_target(array $dnsList, int $active): ?string {
    // Prefer active if enabled
    foreach ($dnsList as $row) {
        if (!is_array($row)) continue;
        $id = (int)($row['id'] ?? 0);
        $en = (bool)($row['enabled'] ?? false);
        $url = trim((string)($row['url'] ?? ''));
        if ($id === $active && $en && $url !== '') return $url;
    }
    // Otherwise first enabled
    foreach ($dnsList as $row) {
        if (!is_array($row)) continue;
        $en = (bool)($row['enabled'] ?? false);
        $url = trim((string)($row['url'] ?? ''));
        if ($en && $url !== '') return $url;
    }
    return null;
}

$base = pick_target($dnsList, $active);
if (!$base) {
    http_response_code(500);
    echo "no enabled dns";
    exit;
}

// Ensure base has trailing slash stripped (we will add our own)
$base = rtrim($base, '/');

$path = (string)($_GET['path'] ?? '');
$path = ltrim($path, '/'); // avoid //

// Build target URL including query string (excluding our internal 'path' param)
$query = $_GET;
unset($query['path']);
$qs = http_build_query($query);

$target = $base . '/' . $path;
if ($qs !== '') $target .= '?' . $qs;

// Prepare curl request
$ch = curl_init($target);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_HEADER, true);
curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 8);
curl_setopt($ch, CURLOPT_TIMEOUT, 20);

// Forward method + body
$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);

$body = file_get_contents('php://input');
if ($method !== 'GET' && $method !== 'HEAD') {
    curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
}

// Forward selected headers
$headers = [];
foreach (getallheaders() as $k => $v) {
    $lk = strtolower($k);
    if (in_array($lk, ['host','content-length','connection','accept-encoding'])) continue;
    $headers[] = $k . ': ' . $v;
}
$headers[] = 'Accept-Encoding: identity';
curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

$resp = curl_exec($ch);
if ($resp === false) {
    http_response_code(502);
    echo "proxy error: " . curl_error($ch);
    curl_close($ch);
    exit;
}

$hdrSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
$rawHeaders = substr($resp, 0, $hdrSize);
$content = substr($resp, $hdrSize);

$code = (int)curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
curl_close($ch);

// Emit response status
http_response_code($code);

// Emit response headers (filter hop-by-hop)
$lines = preg_split("/\r\n|\n|\r/", trim($rawHeaders));
foreach ($lines as $line) {
    if (stripos($line, 'HTTP/') === 0) continue;
    if ($line === '') continue;

    $parts = explode(':', $line, 2);
    if (count($parts) !== 2) continue;
    $name = trim($parts[0]);
    $value = trim($parts[1]);
    $lname = strtolower($name);
    if (in_array($lname, ['transfer-encoding','connection','keep-alive','proxy-authenticate','proxy-authorization','te','trailers','upgrade'])) {
        continue;
    }
    header($name . ': ' . $value, true);
}

echo $content;
